<?php
/**
 * API Client for Tenant Referral Program
 * 
 * Handles all API communications with the referral tracking system
 */

if (!defined('ABSPATH')) {
    exit;
}

class Tenant_Referral_API_Client {
    
    /**
     * Base URL for the API
     */
    private $base_url;
    
    /**
     * API key for authentication
     */
    private $api_key;
    
    /**
     * API endpoint for tracking
     */
    private $track_endpoint;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->base_url = $this->get_base_url();
        $this->api_key = get_option('trp_api_key');
        $this->track_endpoint = rtrim($this->base_url, '/') . TRP_API_ENDPOINT;
        
        if (empty($this->api_key)) {
            add_action('admin_notices', array($this, 'show_api_key_notice'));
        }
    }
    
    /**
     * Get the base URL from settings
     */
    private function get_base_url() {
        $default_url = 'http://localhost:3000';
        
        // Try new plugin option first, fallback to legacy option
        $base_url = get_option('trp_base_url', false);
        if (false === $base_url) {
            $base_url = get_option('inalan_referral_widget_url', $default_url);
        }
        
        return $base_url;
    }
    
    /**
     * Track a referral conversion
     * 
     * @param array $payload The conversion data
     * @return array Result with success status and message
     */
    public function track_conversion($payload) {
        if (empty($this->api_key)) {
            return array(
                'success' => false,
                'message' => 'API key not configured'
            );
        }
        
        if (empty($payload['referralCode'])) {
            return array(
                'success' => false,
                'message' => 'No referral code provided'
            );
        }
        
        // Log the payload for debugging
        error_log("TRP API Client: Tracking conversion with payload: " . print_r($payload, true));
        
        // Prepare request headers
        $request_headers = array(
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $this->api_key,
            'User-Agent' => 'WordPress-TRP-Plugin/' . TRP_VERSION
        );
        
        // Add ngrok headers if needed
        if ($this->is_ngrok_url($this->track_endpoint)) {
            $request_headers['ngrok-skip-browser-warning'] = 'true';
        }
        
        // Prepare request arguments
        $args = array(
            'body'        => json_encode($payload),
            'headers'     => $request_headers,
            'method'      => 'POST',
            'data_format' => 'body',
            'timeout'     => 30, // Increased from 15 to 30 seconds
        );
        
        error_log("TRP API Client: Attempting to POST to URL: " . $this->track_endpoint);
        
        // Make the API request
        $response = wp_remote_post($this->track_endpoint, $args);
        
        // Handle response
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            error_log("TRP API Client: Request failed with error: " . $error_message);
            
            return array(
                'success' => false,
                'message' => 'API request failed: ' . $error_message
            );
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        if ($status_code >= 200 && $status_code < 300) {
            error_log("TRP API Client: Conversion tracked successfully. Response: " . $body);
            
            return array(
                'success' => true,
                'message' => 'Conversion tracked successfully',
                'response' => $body
            );
        } else {
            error_log("TRP API Client: API returned error status " . $status_code . ". Body: " . $body);
            
            return array(
                'success' => false,
                'message' => 'API returned status ' . $status_code,
                'response' => $body
            );
        }
    }
    
    /**
     * Track a landing page visit
     * 
     * @param string $referral_code The referral code
     * @return array Result with success status and message
     */
    public function track_landing_visit($referral_code) {
        if (empty($referral_code)) {
            return array(
                'success' => false,
                'message' => 'No referral code provided'
            );
        }
        
        $payload = array(
            'referralCode' => $referral_code,
            'event_type' => 'LANDING_PAGE_VISIT'
        );
        
        return $this->track_conversion($payload);
    }
    
    /**
     * Build conversion payload from user data
     * 
     * @param string $referral_code The referral code
     * @param int $user_id WordPress user ID
     * @param array $user_data User data array
     * @return array The API payload
     */
    public function build_conversion_payload($referral_code, $user_id, $user_data) {
        return array(
            'referralCode'     => $referral_code,
            'event_type'       => 'CONVERSION',
            'newCustomerEmail' => $user_data['user_email'],
            'newCustomerName'  => $user_data['display_name'] ?: $user_data['user_email'],
            'newCustomerIdWp'  => (string)$user_id,
            'newCustomerPhone' => $user_data['phone_number'] ?? '',
            'newCustomerPostCode' => $user_data['post_code'] ?? '',
            'newCustomerAddress' => $user_data['address'] ?? ''
        );
    }
    
    /**
     * Check if URL is an ngrok URL
     * 
     * @param string $url The URL to check
     * @return bool True if ngrok URL
     */
    private function is_ngrok_url($url) {
        return (strpos($url, 'ngrok-free.app') !== false || strpos($url, 'ngrok.io') !== false);
    }
    
    /**
     * Get the track API URL
     * 
     * @return string The full track API URL
     */
    public function get_track_api_url() {
        return $this->track_endpoint;
    }
    
    /**
     * Get the widget URL
     * 
     * @return string The full widget URL
     */
    public function get_widget_url() {
        return rtrim($this->base_url, '/') . TRP_WIDGET_ENDPOINT;
    }
    
    /**
     * Show admin notice when API key is not configured
     */
    public function show_api_key_notice() {
        ?>
        <div class="notice notice-warning">
            <p>
                <strong><?php _e('Tenant Referral Program:', 'tenant-referral-program'); ?></strong> 
                <?php _e('Please configure your API key in', 'tenant-referral-program'); ?>
                <a href="<?php echo admin_url('options-general.php?page=tenant-referral-program'); ?>">
                    <?php _e('Settings → Referral Program', 'tenant-referral-program'); ?>
                </a> 
                <?php _e('to enable referral tracking.', 'tenant-referral-program'); ?>
            </p>
        </div>
        <?php
    }
} 