<?php
/**
 * Utility Functions for Tenant Referral Program
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Get plugin option with fallback to legacy option
 * 
 * @param string $option_name New option name
 * @param string $legacy_option_name Legacy option name
 * @param mixed $default Default value
 * @return mixed Option value
 */
function trp_get_option($option_name, $legacy_option_name = '', $default = false) {
    $value = get_option($option_name, false);
    
    // If new option doesn't exist and legacy option is provided, try legacy
    if (false === $value && !empty($legacy_option_name)) {
        $value = get_option($legacy_option_name, $default);
    }
    
    // If still no value, use default
    if (false === $value) {
        $value = $default;
    }
    
    return $value;
}



/**
 * Check if current page has any referral shortcodes
 * 
 * @return bool True if page has referral shortcodes
 */
function trp_has_shortcodes() {
    global $post;
    
    if (!is_a($post, 'WP_Post')) {
        return false;
    }
    
    $shortcodes = array(
        'tenant_referral_widget',
        'tenant_referral_signup',
        'tenant_referral_signup_form',
        'tenant_referral_test',
        'inalan_referral_widget' // Legacy support
    );
    
    foreach ($shortcodes as $shortcode) {
        if (has_shortcode($post->post_content, $shortcode)) {
            return true;
        }
    }
    
    return false;
}

/**
 * Get referral code from various sources
 * 
 * @return string|null Referral code if found
 */
function trp_get_referral_code() {
    // Check URL parameter first
    if (isset($_GET['ref_code'])) {
        return sanitize_text_field( wp_unslash( $_GET['ref_code'] ) );
    }
    
    // Check cookie
    $cookie_name = get_option('trp_cookie_name', TRP_COOKIE_NAME);
    if (isset($_COOKIE[$cookie_name])) {
        return sanitize_text_field( wp_unslash( $_COOKIE[$cookie_name] ) );
    }
    
    return null;
}

/**
 * Debug logging helper that respects WordPress debug settings and development mode
 * 
 * @param string $message Log message
 * @param string $prefix Optional prefix (default: 'TRP')
 */
function trp_debug_log($message, $prefix = 'TRP') {
    // Only log if WP_DEBUG is enabled OR development mode is active
    if ((defined('WP_DEBUG') && WP_DEBUG) || get_option('trp_development_mode', false)) {
        error_log($prefix . ': ' . $message);
    }
}

/**
 * Sanitize referral code
 * 
 * @param string $code Referral code to sanitize
 * @return string Sanitized code
 */
function trp_sanitize_referral_code($code) {
    // Remove any non-alphanumeric characters except dashes and underscores
    return preg_replace('/[^a-zA-Z0-9_-]/', '', $code);
} 