<?php
/**
 * Assets Management for Tenant Referral Program
 * 
 * Handles CSS and JavaScript enqueuing with proper dependency management
 * and configuration passing to JavaScript
 */

if (!defined('ABSPATH')) {
    exit;
}

class Tenant_Referral_Assets {
    
    private $plugin_name;
    private $version;
    private $has_shortcode = false;
    
    /**
     * Constructor
     */
    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        
        // Early shortcode detection for conditional loading
        add_action('wp', array($this, 'detect_shortcodes_early'));
        
        // Enqueue assets for frontend (conditionally)
        add_action('wp_enqueue_scripts', array($this, 'enqueue_public_assets'));
        
        // Also enqueue assets for admin when needed (for shortcode testing)
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets_conditional'));
        
        // Add Cloudflare-specific script attributes
        add_filter('script_loader_tag', array($this, 'add_cloudflare_attrs'), 10, 3);
    }
    
    /**
     * Early shortcode detection on 'wp' hook for conditional asset loading
     */
    public function detect_shortcodes_early() {
        global $post;
        
        // Check main post content
        if (is_a($post, 'WP_Post')) {
            // Check for our shortcodes in post content
            if (has_shortcode($post->post_content, 'tenant_referral_widget') ||
                has_shortcode($post->post_content, 'inalan_referral_widget') ||
                has_shortcode($post->post_content, 'tenant_referral_signup') ||
                has_shortcode($post->post_content, 'tenant_referral_signup_form') ||
                has_shortcode($post->post_content, 'tenant_referral_test')) {
                $this->has_shortcode = true;
                $this->debug_log('Shortcode detected in post content: ' . $post->ID);
                return;
            }
        }
        
        // Scan widget instances for our shortcodes to avoid false positives
        $widgets_with_text = (array) get_option('widget_text', array());
        $widgets_with_html = (array) get_option('widget_custom_html', array());
        $candidates = array();
        foreach ($widgets_with_text as $w) { if (isset($w['text'])) { $candidates[] = $w['text']; } }
        foreach ($widgets_with_html as $w) { if (isset($w['content'])) { $candidates[] = $w['content']; } }
        $shortcodes = apply_filters('trp_shortcodes', array(
            'tenant_referral_widget',
            'inalan_referral_widget',
            'tenant_referral_signup',
            'tenant_referral_signup_form',
            'tenant_referral_test'
        ));
        foreach ($candidates as $markup) {
            foreach ($shortcodes as $sc) {
                if (has_shortcode($markup, $sc)) {
                    $this->has_shortcode = true;
                    $this->debug_log('Shortcode detected in widget content');
                    return;
                }
            }
        }
        
        // Force loading in admin context or when explicitly requested
        if (is_admin() || isset($_GET['trp_force_assets'])) {
            $this->has_shortcode = true;
            $this->debug_log('Assets forced in admin context or by request parameter');
        }
    }
    
    /**
     * Check if assets should be loaded
     */
    private function should_load_assets() {
        // Always load in admin when editing posts
        if (is_admin()) {
            return true;
        }
        
        // Load if shortcodes detected
        if ($this->has_shortcode) {
            return true;
        }
        
        // Load if development mode is enabled
        if (defined('WP_DEBUG') && WP_DEBUG && get_option('trp_development_mode', false)) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Enqueue public-facing assets (conditionally)
     */
    public function enqueue_public_assets() {
        // Only load assets when needed
        if (!$this->should_load_assets()) {
            $this->debug_log('Assets not loaded - no shortcodes detected');
            return;
        }
        
        // Check if TRP_PLUGIN_URL is defined
        if (!defined('TRP_PLUGIN_URL')) {
            $this->debug_log('ERROR - TRP_PLUGIN_URL is not defined!');
            return;
        }
        
        $this->debug_log('Loading assets - TRP_PLUGIN_URL: ' . TRP_PLUGIN_URL);
        // Always enqueue CSS (it's lightweight)
        wp_enqueue_style(
            $this->plugin_name . '-public',
            TRP_PLUGIN_URL . 'assets/css/public-styles.css',
            array(),
            $this->version,
            'all'
        );
        
        // Enqueue JavaScript with jQuery dependency
        wp_enqueue_script(
            $this->plugin_name . '-public',
            TRP_PLUGIN_URL . 'assets/js/public-scripts.js',
            array('jquery'),
            $this->get_cache_version(), // Stable version instead of time()
            true // Load in footer
        );
        
        $this->debug_log('Enqueuing script: ' . TRP_PLUGIN_URL . 'assets/js/public-scripts.js');
        
        // Localize script with configuration data
        $this->localize_script_config();
    }
    
    /**
     * Pass configuration data to JavaScript
     */
    private function localize_script_config() {
        // Get plugin settings
        $base_url = get_option('trp_base_url', '');
        $cookie_name = get_option('trp_cookie_name', TRP_COOKIE_NAME);
        $cookie_expiry = get_option('trp_cookie_expiry_days', TRP_COOKIE_EXPIRY_DAYS);
        
        // Build API URLs
        $track_api_url = '';
        if (!empty($base_url)) {
            $track_api_url = rtrim($base_url, '/') . TRP_API_ENDPOINT;
        }
        
        // Prepare configuration array
        $config = array(
            'cookieName' => $cookie_name,
            'cookieExpiry' => intval($cookie_expiry),
            'trackApiUrl' => $track_api_url,
            'baseUrl' => $base_url,
            'widgetEndpoint' => TRP_WIDGET_ENDPOINT,
            'apiEndpoint' => TRP_API_ENDPOINT,
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('trp_ajax_nonce'),
            'debug' => (defined('WP_DEBUG') && WP_DEBUG) || get_option('trp_development_mode', false),
            'version' => $this->version
        );
        
        // Add legacy support
        $legacy_url = get_option('inalan_referral_widget_url', '');
        if (!empty($legacy_url) && empty($base_url)) {
            $config['trackApiUrl'] = rtrim($legacy_url, '/') . TRP_API_ENDPOINT;
            $config['baseUrl'] = $legacy_url;
        }
        
        // Localize the script
        wp_localize_script(
            $this->plugin_name . '-public',
            'TRP_Config',
            $config
        );
        
        // Add inline script for debugging if WP_DEBUG is enabled
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $debug_script = "
                console.log('TRP: Configuration loaded', TRP_Config);
                console.log('TRP: Plugin version " . $this->version . "');
                console.log('TRP: JavaScript URL should be: " . TRP_PLUGIN_URL . "assets/js/public-scripts.js');
            ";
            wp_add_inline_script($this->plugin_name . '-public', $debug_script);
        }
    }
    
    /**
     * Enqueue admin assets conditionally (for shortcode testing in admin)
     */
    public function enqueue_admin_assets_conditional($hook) {
        // Check if we're on a page that might have shortcodes
        global $post;
        
        $should_load = false;
        
        // Load on our settings page
        if ($hook === 'settings_page_tenant-referral-program') {
            $should_load = true;
        }
        
        // Load on post edit pages that might contain shortcodes
        if (in_array($hook, array('post.php', 'post-new.php', 'page.php', 'page-new.php'))) {
            $should_load = true;
        }
        
        // Load if current page content contains our shortcodes
        if (is_a($post, 'WP_Post')) {
            if (has_shortcode($post->post_content, 'tenant_referral_test') ||
                has_shortcode($post->post_content, 'tenant_referral_widget') ||
                has_shortcode($post->post_content, 'tenant_referral_signup') ||
                has_shortcode($post->post_content, 'tenant_referral_signup_form')) {
                $should_load = true;
            }
        }
        
        if ($should_load) {
            // Enqueue the same public assets for admin context
            $this->enqueue_public_assets();
        }
    }
    
    /**
     * Enqueue admin assets (for settings page)
     */
    public function enqueue_admin_assets($hook) {
        // Only load on our settings page
        if ($hook !== 'settings_page_tenant-referral-program') {
            return;
        }
        
        wp_enqueue_style(
            $this->plugin_name . '-admin',
            TRP_PLUGIN_URL . 'assets/css/admin-styles.css',
            array(),
            $this->version,
            'all'
        );
        
        wp_enqueue_script(
            $this->plugin_name . '-admin',
            TRP_PLUGIN_URL . 'assets/js/admin-scripts.js',
            array('jquery'),
            $this->version,
            true
        );
    }
    
    /**
     * Check if shortcodes are present (for conditional loading)
     */
    public function has_referral_shortcodes() {
        return $this->has_shortcode;
    }
    
    /**
     * Force enqueue assets (for programmatic use)
     */
    public function force_enqueue_assets() {
        $this->has_shortcode = true;
        $this->enqueue_public_assets();
    }
    
    /**
     * Get cache version for assets
     */
    private function get_cache_version() {
        // Use time() only in development/debug mode, otherwise use static version
        if (defined('WP_DEBUG') && WP_DEBUG && get_option('trp_development_mode', false)) {
            return $this->version . '-' . time();
        }
        return $this->version;
    }
    
    /**
     * Debug logging helper (respects WP_DEBUG setting)
     */
    private function debug_log($message) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('TRP: ' . $message);
        }
    }
    
    /**
     * Add Cloudflare-specific attributes to prevent script reordering
     */
    public function add_cloudflare_attrs($tag, $handle, $src) {
        if ($handle === $this->plugin_name . '-public') {
            // Prevent Cloudflare Rocket Loader from reordering this script
            return str_replace('<script', '<script data-cfasync="false"', $tag);
        }
        return $tag;
    }
    
    /**
     * Get asset URLs for external use
     */
    public function get_asset_urls() {
        return array(
            'css' => TRP_PLUGIN_URL . 'assets/css/public-styles.css',
            'js' => TRP_PLUGIN_URL . 'assets/js/public-scripts.js',
            'version' => $this->version
        );
    }
} 